#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Joseph Elmes : NERC Funded PhD Researcher in Applied Mathematics
University of Leeds : Leeds LS2 9JT : ml14je@leeds.ac.uk

Python 3.7 : Wed Aug  7 12:13:58 2019
"""

def plot_setup(x_label='', y_label='', x_log=False, y_log=False, scale=1, 
               title='', dpi=100):
    """
    Sets up figure environment
    """
    import matplotlib.pyplot as pt
    import matplotlib as mpl

    mpl.style.use('seaborn-ticks')
    fig, ax = pt.subplots(figsize=(13.68*scale, 7.68*scale), dpi=dpi)
    pt.xticks(fontsize=16)
    pt.yticks(fontsize=16)
    pt.xlabel(x_label, fontsize=20)
    pt.ylabel(y_label, fontsize=20)
    pt.title(title, fontsize=22)
    if x_log: pt.xscale('log')
    if y_log: pt.yscale('log')
    pt.grid(linewidth=0.5)

    return fig, ax

def Chebyshev_grid(a, b, n):
    """
    Creates a 1D grid on [a, b] corresponding to n+1 Chebyshev points
    """
    import numpy as np
    theta, middle, radius = np.pi/n, (a+b)/2, (b-a)/2

    return np.array([middle-radius*np.cos(i*theta) for i in range(n+1)])

def T_n(n, y):
    """
    Returns the first n+1 evaluations of Chebyshev polynomials of the first
    kind on y.
    """
    import numpy as np

    T = np.zeros((1, len(y)))
    
    for i in range(n+1):
        if i == 0:
            T[0,:] = 1
        elif i == 1:
            T = np.append(T, [y], axis=0)
        else:
            T = np.append(T, [2*y*T[-1] - T[-2]], axis=0)

    return T

def T_n_func(n):
    """
    Returns the numpy function of the n-th Chebyshev polynomial of the first
    kind.
    """

    import numpy as np
    Tn_coeff = np.zeros(n+1)
    Tn_coeff[-1] = 1
    
    return np.polynomial.chebyshev.Chebyshev(Tn_coeff)

def U_n(n, y):
    """
    Returns the numpy function of the n-th Chebyshev polynomial of the second
    kind.
    """

    import scipy
    import numpy as np
    Un_coeff = np.zeros(n+1)
    Un_coeff[-1] = 1
    
    return scipy.special.eval_chebyu(Un_coeff)

def p_n(coeff_vec, x):
    
    S = 0
    for i, a in enumerate(coeff_vec):
        S+=a*(x**i)
    
    return S

def p_n_der(coeff_vec, x):
    
    S = 0
    for i, a in enumerate(coeff_vec):
        if i != 0:
            S+=i*a*(x**(i-1))

    return S

def p2_norm(v):
    """
    Evaluates the p2 norm of vector v.
    """
    import numpy as np
    return np.sqrt(np.matmul(v, v))

def pInf_norm(v):
    """
    Evaluates the p_infinity norm of vector v.
    """
    import numpy as np
    return np.max(np.abs(v))

def trapezium_rule(f, a, b, n):
    """
    Integrates the function f between a and b with n + 1 grid points using the
    trapezium rule.
    """
    S, h = (f(a)+f(b))/2, (b-a)/n

    for i in range(1, n):
        S+=f(a+i*h)
    
    return h*S

def newton_raphson(f, init_val, sup_output=True, max_iteration=250):
    """
    Newton--Raphson method which utilises the Secant-Method to
    approximate the derivative of function f. The result should return the#
    closest root of f to init_val.
    """
    tolerance, F, eps = 1E-15, f(init_val), 1E-8
    error, iteration = abs(F), 1
    val = init_val

    while error > tolerance:
        f_prime = (f(val+eps/2)-f(val-eps/2))/eps

        val -= F/f_prime
        F = f(val)
#        print(F)
        error = abs(F)
        iteration +=1

        if iteration > max_iteration:
#            print(F)
            if not sup_output:
                print('Could not converge to machine precision within {} \
#iterations.\n'.format(max_iteration))
            return val
    
    return val

def x_to_t(x_val, a, b):
    return (2*x_val-(b+a))/(b-a)

def t_to_x(t_val, a, b):
    return ((b-a)*t_val+(b+a))/2

def root_sqr_mean_err(f1, f2, a, b):
    """
    Returns the weighted root-mean-square error.
    """
    import numpy as np
    from math import pi
    g = lambda theta: (f1(np.cos(theta))-f2(np.cos(theta)))**2
    I = trapezium_rule(g, 0, pi, int(5E3))
    return np.sqrt(I)/np.sqrt(b-a)