#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
COPYRIGHT NOTICE: THIS SCRIPT IS FOR THE SOLE USE OF STUDENTS 
ATTENDING THE MATH3474 COURSE. ANY PERSON WISHING TO COPY, USE OR 
PROCESS THIS MATERIAL IN ANY FORM FOR ANY OTHER PURPOSES SHOULD 
CONTACT THE AUTHOR VIA EMAIL.

THIS SCRIPT WAS ORIGINALLY PRODUCED IN MAPLE BY PROF MARK KELMANSON, AND
ADAPTED INTO PYTHON BY AUTHOR.

Joseph Elmes : NERC-Funded PhD Researcher in Applied Mathematics
University of Leeds : Leeds LS2 9JT : ml14je@leeds.ac.uk

Python 3.7 : Fri Aug 23 12:44:16 2019

MATH3474 : Section 3 : Lectures 19 to 20
SCRIPT MATH3474_8.py : ITERATION MATRICES and CONVERGENCE RATES
"""

import numpy as np
import matplotlib.pyplot as pt
from MATH3474 import plot_setup
from MATH3474_6 import rho, two_norm

def find_min(f, x1, x2, d=16):
    """
    Returns the minimum of function f on domain [x1, x2] to d digits of
    precision.
    
    Parameters
    ----------
    f : Function
        Function which we try to minimise on domain [x1, x2].
    x1 : Float
        Lower limit of domain on f(x).
    x2 : Float
        Upper limit of domain on f(x).
    d : Integer
        Digits of precision for simple minimising function. Default is 16,
        machine precision.
    """
    i = 1
    while i < d:
        dx = 10**-i
        x_vals = np.arange(x1, x2+dx, dx)
        y_vals = np.zeros(len(x_vals))
        for j, x in enumerate(x_vals):
            y_vals[j] = rho(f(x))
        x0 = x_vals[np.argmin(y_vals)]

        if x0==x1 or x0==x2:
            break

        x1, x2 = x0-dx, x0+dx
        i+=1
        
    return x0

def diag_rescale(M):
    """
    Rescales each row to ensure the row's diagonal element is 1.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we return the diagonal re-scaling.
    """
    n, m = M.shape
    assert n == m #We require M to be a square matrix
    M_new = np.copy(M)
    for i in range(n):
        Mii = M[i, i]
        M_new[i, :] /= Mii 
    
    return M_new

def L_decomp(M):
    """
    Decomposes square matrix M and returns its lower triagonal.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we return the L-decomposition.
    """
    n, m = M.shape
    assert n == m #We require M to be a square matrix
    L = np.zeros((n, n))
    
    for i in range(n):
        for j in range(n):
            if j<i:
                L[i, j] = M[i, j]
                
    return -L

def U_decomp(M):
    """
    Decomposes square matrix M and returns its upper triagonal.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we return the U-decomposition.
    """
    n, m = M.shape
    assert n == m #We require M to be a square matrix
    U = np.zeros((n, n))
    
    for i in range(n):
        for j in range(n):
            if j>i:
                U[i, j] = M[i, j]
                
    return -U

def Jacobi(M):
    """
    Returns the Jacobi matrix of square matrix M.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we return the Jacobi matrix.
    """
    B = diag_rescale(M)
    L, U = L_decomp(B), U_decomp(B)
    
    return L + U

def Gauss_Seidel(M):
    """
    Returns the Guass-Seidel matrix of square matrix M.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we return the Gauss-Seidel matrix.
    """
    from numpy.linalg import inv
    n, m = M.shape
    assert n == m #We require M to be a square matrix
    B, I = diag_rescale(M), np.eye(n) #np.eye(n) returns I_n
    L, U = L_decomp(B), U_decomp(B)

    return np.matmul(inv(I-L), U)

def SOR(M, ω):
    """
    Returns the SOR matrix of square matrix M with SOR parameter ω.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we return the SOR matrix.
    ω : Float
        SOR parameter which typically lies in the domain [1, 2].
    """
    from numpy.linalg import inv
    n, m = M.shape
    assert n == m #We require M to be a square matrix
    B, I = diag_rescale(M), np.eye(n) #np.eye(n) returns I_n
    L, U = L_decomp(B), U_decomp(B)
    
    return np.matmul(inv(I-ω*L), (1-ω)*I+ω*U)

def SOR_r_vals(M, ω_vals):
    """
    Returns the spectral radius of the SOR matrix for a given matrix M and
    ω for ω in parameter array ω_vals.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we return the SOR matrix.
    ω_vals : numpy.ndarray
        Values of SOR parameter for which we consider returning the spectral
        radius of matrix M's SOR matrix.
    """
    r_vals = np.zeros(len(ω_vals))
    for i, ω in enumerate(ω_vals):
        r_vals[i] = rho(SOR(M, ω))
        
    return r_vals

def is_two_cyclic(M):
    """
    Determines whether matrix M is two-cyclic by returning either True or False.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix which we determine is two-cyclic.
   """
    n, m = M.shape
    assert n == m #We require M to be a square matrix

    A = diag_rescale(M)
    for i in range(n):
        if (np.all(A[:i, :i]==np.eye(i)) and\
            np.all(A[i:, i:]==np.eye(n-i))):
            return True

    return False

def Example_3(M):
    """
    Example 3.3:
    Compares the Jacobi and Gauss-Seidel methods, noting that that if the
    spectral radius of the iterative matrix is greater than 1, the scheme is
    unstable, and stable closer to 0.

    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we compare the Jacobi and Gauss-Seidel methods
        when solving for x in Mx=b.
    """
    n, m = M.shape
    assert n == m #We require M to be a square matrix
    H_J, H_GS = Jacobi(M), Gauss_Seidel(M)
    
    for mat, lab in zip([H_J, H_GS], ['Jacobi', 'Gauss-Seidel']):
        print('{}:\n'.format(lab))
        for i in range(n):
            print('\t', mat[i])
            
        print('\n\tSpectral radius: {}'.format(round(rho(mat), 4)))
        print('\tTwo-norm: {}\n\n'.format(round(two_norm(mat), 4)))

def Example_4(M):
    """
    Example 3.4:
    Compares the Jacobi, Gauss-Seidel and SOR schemes. We detect also if matrix
    M is two-cyclic and symmetrical, and if so, verifies equation (3.19) in
    notes.

    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we compare the Jacobi,Gauss-Seidel and SOR 
        methods when solving for x in Mx=b.
    """
    n, m = M.shape
    assert n == m #We require M to be a square matrix

    fig, ax = plot_setup('$\\omega$', '$\\rho(H_{\\omega})$',
                         scale=s)
    ω_vals = np.linspace(1, 2, 500)
    ax.plot(ω_vals, SOR_r_vals(M, ω_vals))
    pt.show()

    g = lambda x: SOR(M, x)
    ω = find_min(g, 1, 2)
    H_J, H_GS, H_ω = Jacobi(M), Gauss_Seidel(M), SOR(M, ω)
    r_J, r_GS, r_ω = rho(H_J), rho(H_GS), rho(H_ω)
    labels = ['Jacobi', 'Gauss-Seidel', 'SOR for ω={}'.format(round(ω, 6))]
    for mat, label, r in zip([H_J, H_GS, H_ω], labels, [r_J, r_GS, r_ω]):
        print('{}:\n'.format(label))
        for i in range(n):
            print('\t', mat[i])
        print('\n\tSpectral radius: {}'.format(round(r, 5)))
        print('\tTwo-norm: {}\n\n'.format(round(two_norm(mat), 5)))

    print('Efficiency factors:')
    print('\tSOR_to_J = {}'.format(round(np.log(r_ω)/np.log(r_J), 4)))
    print('\tSOR_to_GS = {}'.format(round(np.log(r_ω)/np.log(r_GS), 4)))
    print('\tGS_to_J = {}\n\n'.format(round(np.log(r_GS)/np.log(r_J), 4)))

    if np.all(M == M.T) and is_two_cyclic(M):
        print('Matrix is both two-cyclic and symmetric:')
        ω_opt = 2/(1+np.sqrt(1-r_J**2)) # eqn (3.19) in notes for ω*
        H_ω_opt = SOR(M, ω_opt)
        r_ω_opt = rho(H_ω_opt)
        print('\tω*={} using eqn (3.19) from notes'.format(round(ω_opt, 7)))
        print('\tSpectral radius of H_ω*: {}'.format(round(r_ω_opt, 7)))
        print('\tρ(H_{ω∗}) = ω∗−1:', round(r_ω_opt, 7)==round(ω_opt-1, 7))
        
def Example_5():
    """
    Example 3.5:
    Displays the optimum SOR parameter for 2-cyclic matrices using equation 
    while varying spectral radius using equations (3.24) and (3.28) in notes.
    """
    
    def top(ω, λ):
        return (ω*λ/2 + np.sqrt((ω**2*(λ**2))/4-ω+1))**2
        
    def bottom(ω, λ):
        return (ω*λ/2 - np.sqrt((ω**2*(λ**2))/4-ω+1))**2

    fig, ax = plot_setup('$\\omega$', '$|\\mu|$', scale=s)
    ω = np.linspace(0, 2, int(1E5))
    ρ_vals = np.array([0.1, 0.3, 0.5, 0.7, 0.9, 0.99, 0.999])

    for i, ρ in enumerate(ρ_vals):
        ax.plot(ω, top(ω, ρ), 'r-', linewidth=s)
        ax.plot(ω, bottom(ω, ρ), 'b-', linewidth=s)
    
    ax.plot([1, 2], [0, 1], 'g-', linewidth=2*s)
    pt.show()


if __name__ == '__main__':
    s = .5 #scale of plots on screen

    A1 = np.array([
            [3, 1, 2],
            [-1, 3, -2],
            [-2, 2, 3]
            ], dtype=float)
    
#    #Example 3.2: Convergence affected by re-ordering equations
#    Example_3(A1)
    
    A2 = np.array([
            [-4, 1, 1, 1],
            [1, -4, 1, 1],
            [1, 1, -4, 1],
            [1, 1, 1, -4]
            ], dtype=float)
    
    A3 = np.array([
            [-4, 0, 1, 1],
            [0, -4, 1, 1],
            [1, 1, -4, 0],
            [1, 1, 0, -4]
            ], dtype=float)

#    #Example 3.4: SOR Method
#    Example_4(A2)
#    Example_4(A3)
    
#    #Example 3.5: The optimum SOR parameter for 2-cyclic matrices
#    Example_5()