#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
COPYRIGHT NOTICE: THIS SCRIPT IS FOR THE SOLE USE OF STUDENTS 
ATTENDING THE MATH3474 COURSE. ANY PERSON WISHING TO COPY, USE OR 
PROCESS THIS MATERIAL IN ANY FORM FOR ANY OTHER PURPOSES SHOULD 
CONTACT THE AUTHOR VIA EMAIL.

THIS SCRIPT WAS ORIGINALLY PRODUCED IN MAPLE BY PROF MARK KELMANSON, AND
ADAPTED INTO PYTHON BY AUTHOR.

Joseph Elmes : NERC-Funded PhD Researcher in Applied Mathematics
University of Leeds : Leeds LS2 9JT : ml14je@leeds.ac.uk

Python 3.7 : Fri Aug 16 19:45:52 2019

MATH3474 : Section 2 : Lectures 12 to 13
SCRIPT MATH3474_4.py : FINITE-DIFFERENCE OPERATORS
"""

import numpy as np
import matplotlib.pyplot as pt
from MATH3474 import plot_setup
from math import pi

def fin_diff_scheme(grid, d):
    """
    Calculates the weight coefficients for a particular numpy integer grid of
    the form [x_{i-n}, ..., x_i, ..., x_{i+m}] corresponding to the input of
    np.array([-n, -n+1, ..., 0, ..., m-1, m]) for the d-order derivative. As 
    well as the coefficients of the grid points, the function also returns the
    coefficient of the next order term, for which it will be a product of the 
    next derivative evaluated at x_i.
    
    Parameters
    ----------
    grid : numpy.ndarray
        Indicates the grid of points we use to calculate the weights of the
        finite-difference.
    d : Integer
        Order of the derivative we approximate.
    """
    from math import factorial
    
    N = len(grid)
    if not N>d: #ensures that the number of grid points is greater than the
                #order of the derivate.
        print("Grid size (N) must be greater than the order of the derivative \
(d) being approximated. I.e, we require N > d.")
        return
    
    M = np.zeros((N, N))
    
    for i in range(N):
        M[i] = grid**i

    v = np.zeros(N)
    v[d] = factorial(d)
    
    weights = np.matmul(np.linalg.inv(M), v)
    err = np.matmul(grid**N, weights)/factorial(N)

    return weights, err

def der_calc(f, x0, dx, grid, d):
    """
    Returns the evaluation of the d-th derivative approximation at x=x0 using
    the stencil, 'grid', with the various h values, dx. Note that 'grid' is a
    1D array of integers, while dx is a 1D arrray of floats.    

    Parameters
    ----------
    f : Function
        Function of which we are approximating the d-order derivative.
    x0 : Float
        The value for which we are evaluating the d-order derivative of f(x).
    dx : Float
        Indicates the uniform grid spacing between each finite-difference
        interval x_{i+1}-x_{i}.
    grid : numpy.ndarray
        Indicates the grid of points we use to calculate the weights of the
        finite-difference.
    d : Integer
        Order of the derivative we approximate.
    """
    coeffs, error = fin_diff_scheme(grid, d)
    grid = grid.reshape(len(grid), 1)
    x_vals = x0 + np.matmul(dx, grid.T)
    f_vals = f(x_vals)

    S = 0
    for i in range(len(grid)):
        S += coeffs[i]*f_vals[:, i]

    for i in range(len(dx)):
        S[i] /= dx[i]**d

    return S

def Example_1(grid, d):
    """
    Example 2.1:
    For a given grid in the form of a 1D array of integers, the function
    prints the weights for the grid points. The function will also print the
    error term.

    Parameters
    ----------
    grid : numpy.ndarray
        Indicates the grid of points we use to calculate the weights of the
        finite-difference.
    d : Integer
        Order of the derivative we approximate.
    """
    
    N = len(grid)
    weights, error = fin_diff_scheme(grid, d)
    
    print('Weights:\n')
    
    for i in range(len(grid)):
        print('\tx_{{{0}}}: w_{{{0}}}={1}'.format(grid[i], round(weights[i],
              4)))
        
    print('\nError Term: {}h^{{{}}} f^{{({})}}(x_0)'.format(round(error, 4),
          N-d, d+1))    

def Example_2(option, d, f_symb, x0):
    """
    Example 2.2:
    Given the choice between central difference (C), backward difference (B)
    and forward difference (F), we compare the d-th derivative of f_symb(x)
    evaluated at x = x0.

    Parameters
    ----------
    option : String
        Indicates whether finite-difference scheme is central, forward or 
        backward.
    d : Integer
        Order of the derivative we approximate.
    f_symb : sympy.core.function
        The symbolic function of that which we are approximating the d-order
        derivative.
    x0 : Float
        The value at which we evaluate and compare the d-order derivative of
        f(x).
    """
    from sympy import lambdify, diff, latex
    
    f = lambdify(x, f_symb, 'numpy')
    dx = 10**np.linspace(-4, 0, 49)
    true_val = lambdify(x, diff(f_symb, x, d), 'numpy')(x0) #exact value
    dx = dx.reshape((len(dx), 1)) #different dx values to plot
    orders = np.linspace(1, 6, 6, dtype=int) #possible finite-difference order
                                             #schemes we consider
    grids=[] # list to store grids, in the form of arrays, of various lengths

    if option.upper() == 'C': #Central Difference
        orders = orders[1:][::2] #central difference schemes are of even order
        for o in orders:
            m = int((d+o-1-(d-1)%2)/2)
            grid = np.linspace(-m, m, 2*m+1, dtype=int)
            grids.append(grid)
    
    elif option.upper() == 'F': #Forward Difference
        for o in orders:
            grid = np.linspace(0, o+d-1, o+d, dtype=int)
            grids.append(grid)
            
    elif option.upper() == 'B': #Backward Difference
        for o in orders:
            grid = np.linspace(-(o+d-1), 0, o+d, dtype=int)
            grids.append(grid)
    
    else:
        print('Option is not valid. End Example 2.2.')
        return

    #Plot absolute errors:
    title = '$f(x)={}$ at $x_0={}$'.format(latex(f_symb),
                round(x0, 4)) #plot title
    fig, ax = plot_setup('$\\Delta x$',
        'Absolute Error on $f^{{({})}}(x_0)$'.format(d),
    x_log=True, y_log=True, scale=s, title=title) #set up plotting environ
    for i, grid in enumerate(grids):
        D = der_calc(f, x0, dx, grid, d) #calculate the approximation
        ax.plot(dx, np.abs(D-true_val), label = orders[i]) #plot absolute error
        ax.plot(dx, dx**orders[i], 'k:')
    ax.legend(fontsize=s*16)
    pt.show()

if __name__ == '__main__':
    s = .5 #scale of plots on screen
    from sympy import sin, cos, tan, exp, sqrt
    from sympy.abc import x
    
    d = 1 #order of derivative

#    #Example 2.1: Finite-Difference Coefficients
#    grid = np.array([0, 1, 2, 3])
#    Example_1(grid, d)
#
#    #Example 2.2: Finite-Difference Scheme Analysis
#    f_expr = exp(cos(x)) # Choose function, f(x), symbolically
#    x0 = 2*pi/3 # Choose x-value to evaluate f^(n)(x)
#    option = 'C' # C - central, B - Backward, F - Forward
#    Example_2(option, d, f_expr, x0) #Compare f^(n)(x_0) up to sixth order