#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
COPYRIGHT NOTICE: THIS SCRIPT IS FOR THE SOLE USE OF STUDENTS 
ATTENDING THE MATH3474 COURSE. ANY PERSON WISHING TO COPY, USE OR 
PROCESS THIS MATERIAL IN ANY FORM FOR ANY OTHER PURPOSES SHOULD 
CONTACT THE AUTHOR VIA EMAIL.

THIS SCRIPT WAS ORIGINALLY PRODUCED IN MAPLE BY PROF MARK KELMANSON, AND
ADAPTED INTO PYTHON BY AUTHOR.

Joseph Elmes : NERC-Funded PhD Researcher in Applied Mathematics
University of Leeds : Leeds LS2 9JT : ml14je@leeds.ac.uk

Python 3.7 : Thu Aug 8 11:48:33 2019

MATH3474 : Section 1 : Lectures 5 to 6
SCRIPT MATH3474_2.py : MINIMAX APPROXIMATION
"""

import numpy as np
import matplotlib.pyplot as pt
from MATH3474 import plot_setup
from MATH3474_1 import lagrange_interp

def N_dim_newton_raphson(F, vals, tol=1E-16, max_iteration=500):
    """
    N-dimensional Newton--Raphson method which utilises the Secant-Method to
    approximate the Jacobian matrix. The result should yield roots of F(X)=0,
    where X = vals.
    
    Parameters
    ----------
    F : Function
        Vector function of which we attempt to solve roots F(X)=0.
    vals : numpy.ndarray
        Array of initial guesses for the roots of F(X)=0.
    tol : Float
        Error tolerance for p2 norm of F(X).
    max_iteration : Integer
        Maximum number of iterations for the Newton--Raphson method.
    """
    from MATH3474 import p2_norm
    from numpy.linalg import inv, LinAlgError

    error, iteration  = p2_norm(F(vals)), 1

    while error > tol:
        J = Jacobian_matrix(F, vals)

        try:
            J_inv = inv(J)
        except LinAlgError as err:
            if 'Singular matrix' in str(err):
                print('Singular Matrix. Return result.\n')
                return vals

        vals -= np.matmul(J_inv, F(vals))
        error = p2_norm(F(vals))

        if iteration > max_iteration:
            print('Could not converge to machine precision within {} \
iterations.\n'.format(max_iteration))
            return vals
        
        iteration +=1
    
    return vals

def Jacobian_matrix(F, vals, eps=1E-12):
    """
    Generates the Jacobian matrix of vector function F around variable values
    var.
    
    Parameters
    ----------
    F : Function
        Vector function of which one constructs the Jacobian matrix.
    vals : numpy.ndarray
        Vector values at which one constructs the Jacobian matrix.
    eps : Float
        The small change in the vector values to approximate the partial
        derivatives in the constructed Jacobian matrix. The default value is
        1E-12.
    """
    
    m = len(vals)
    M = np.zeros((m, m))
    for i in range(m):
        dvals1, dvals2 = np.copy(vals), np.copy(vals)
        dvals1[i] += eps/2
        dvals2[i] -= eps/2
        M[:, i] = (F(dvals1)-F(dvals2))/eps

    return M

def minimax_func(f, f_prime, var, a, b):
    """
    Returns the vector function necessary to equal zero when finding the
    minimax polynomial.
    
    Parameters
    ----------
    f : Function
        The function of which we are attempting to construct a minimax
        interpolation polynomial.
    f_prime : Function
        The derivative of the function of which we are attempting to construct
        a minimax interpolation polynomial.
    var : numpy.ndarray
        The variable array of values of which we evaluate the vector function.
        In theory, for our minimax approximation, we should expect these, if
        correct, to be roots of the vector function.
    a : Float
        The lower limit of the domain over which we construct the minimax
        polynomial.
    b : Float
        The upper limit of the domain over which we construct the minimax
        polynomial.
    """
    from MATH3474 import p_n, p_n_der

    n, F = int(len(var)/2-1),  np.zeros(len(var))
    for i in range(n):
        F[i]=f(var[i])-p_n(var[n:-1], var[i])+((-1)**i)*var[-1]
        F[n+i]=f_prime(var[i])-p_n_der(var[n:-1], var[i])
    F[-2] = f(a)-p_n(var[n:-1], a)-var[-1]
    F[-1] = f(b)-p_n(var[n:-1], b)+((-1)**(n))*var[-1]
    
    return F

def minimax(f_sym, a, b, n, sup_output=False):
    """
    Solves minimax system on f_symb, the symbolic representation of f(x), and
    returns parameter values [x_1, ..., x_n, a_0, ..., a_N, rho], where x_i is
    the minimax point, a_i is the coefficient to x^i in the minimax
    interpolation polynomial q_n(x) and rho is the absolute error of q_n on
    points {a, x_1, x_2, ..., x_n, b}.
    
    Parameters
    ----------    
    f_sumb : sympy.core.function
        Symbolic function to which the minimax polynomial is constructed.
    a : Float
        The lower limit of the domain over which we construct the minimax
        polynomial.
    b : Float
        The upper limit of the domain over which we construct the minimax
        polynomial.
    n : Integer
        Indicates the order of the minimax interpolating polynomial.
    sup_output : Boolean
        If True, the function will supress printing the output.
    """
    from sympy import diff, lambdify
    from sympy.abc import x

    var_init = np.zeros(2*(n+1))
    var_init[:n] = np.linspace(a, b, n+2)[1:-1]
    
    f = lambdify(x, f_sym, 'numpy')
    f_prime = lambdify(x, diff(f_sym, x), 'numpy')
    F = lambda init_vals : minimax_func(f, f_prime, init_vals, a, b)
    var = N_dim_newton_raphson(F, var_init)

    if not sup_output:
        print('Minimax points:')
        for i in range(n):
            print('\tx_{} = {}'.format(i+1, round(var[i], 4)))
        func = 0
        for i in range(n+1):
            func += round(var[n+i], 4)*(x**i)
        print('\nMinimax polynomial:\n\tq_{}(x) = {}'.format(n, func))
        print('\nMaximum absolute error on [a, b]:\n\trho\
= {}\n\n'. format(round(var[-1], 4)))

    return var

def Example_9(f, MM_func, a, b, n):
    """
    Example 1.9:
    Compares the minimax interpolating function with a Taylor series expansion
    of the same polynomial order and a regularly-spaced lagrange interpolation
    function.
    
    Parameters
    ----------    
    f : sympy.core.function
        Symbolic function on which we construct interpolating polynomials.
    MM_func : Function
        Minimax interpolating function.
    a : Float
        The lower limit of the domain over which we construct the interpolating
        polynomials.
    b : Float
        The upper limit of the domain over which we construct the interpolating
        polynomials.
    n : Integer
        Indicates the order of the interpolating polynomials we wish to
        compare.
    """
    from sympy import series, lambdify, latex

    X, x_arr = np.linspace(a, b, n+1), np.linspace(a, b, 500)
    g = lambdify(x, f, 'numpy') #function of symbolic expression

    #n+1-term Taylor expansion of f centered at (a+b)/2:
    h = lambdify(x, series(f, x, (a+b)/2, n+1).removeO(), 'numpy')
    
    fig, ax = plot_setup('$x$', scale=s)
    ax.plot(x_arr, g(x_arr), label = '${}$'.format(latex(f)))
    ax.plot(x_arr, lagrange_interp(x_arr, X, g(X)), linestyle=':',
            label='Regularly-spaced Lagrange polynomial')
    ax.plot(x_arr, MM_func(x_arr), linestyle = '--',
            label='Minimax interpolation')
    
    # Value error when h(x), Taylor expansion of f(x), is a constant:
    try:
        ax.plot(x_arr, h(x_arr), linestyle='-.',
                label='$O(x^{{{}}})$ Taylor expansion'.format(n+1))
    except ValueError:
        ax.plot(x_arr, h(x_arr)*np.ones(len(x_arr)), linestyle='-.',
                label='$O(x^{{{}}})$ Taylor expansion about {}'.format(n+1))

    ax.legend(fontsize=16*s)
    pt.show()
    
def Example_10(f, MM_func, a, b, n):
    """
    Example 1.10:
    Compares the minimax interpolating function with a Taylor series expansion
    of the same polynomial order and a regularly-spaced lagrange interpolation
    function.

    Parameters
    ----------    
    f : sympy.core.function
        Symbolic function on which we construct interpolating polynomials.
    MM_func : Function
        Minimax interpolating function.
    a : Float
        The lower limit of the domain over which we construct the interpolating
        polynomials.
    b : Float
        The upper limit of the domain over which we construct the interpolating
        polynomials.
    n : Integer
        Indicates the order of the interpolating polynomials we wish to
        compare.
    """
    from sympy import series, lambdify

    X, x_arr = np.linspace(a, b, n+1), np.linspace(a, b, 500)
    g = lambdify(x, f, 'numpy') #function of symbolic expression

    #n+1-term Taylor expansion of f centered at (a+b)/2:
    h = lambdify(x, series(f, x, (a+b)/2, n+1).removeO(), 'numpy')
    
    fig, ax = plot_setup('$x$', 'Error', y_log=False, scale=s)

    ax.plot(x_arr, g(x_arr)-lagrange_interp(x_arr, X, g(X)), linestyle=':',
            label='Regularly-spaced Lagrange polynomial')
    ax.plot(x_arr, g(x_arr)-MM_func(x_arr),
            linestyle = '--', label='Minimax interpolation')
    ax.plot(x_arr, g(x_arr)-h(x_arr), linestyle='-.',
            label='$O(x^{{{}}})$ Taylor expansion'.format(n+1))
    ax.legend(fontsize=16*s)
    pt.show()

if __name__ == '__main__':
    s = .5 #scale of plots on screen
    from sympy import exp #, cos, sin, tan
    from sympy.abc import x
    from MATH3474 import p_n
    
    a, b, n = -1, 1, 3 #a, b, and n are parameter values as defined in notes
    f_expr = exp(x) # function on x in [a, b] used in the following examples

    MM = minimax(f_expr, a, b, n)
    MM_func = lambda y: p_n(MM[n:-1], y) #This is a lambda function  

    #Example 1.9: Minimax Inteprolation
    Example_9(f_expr, MM_func, a, b, n)
#    
#    #Example 1.10: Minimax Error Comparison
#    Example_10(f_expr, MM_func, a, b, n)