#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
COPYRIGHT NOTICE: THIS SCRIPT IS FOR THE SOLE USE OF STUDENTS 
ATTENDING THE MATH3474 COURSE. ANY PERSON WISHING TO COPY, USE OR 
PROCESS THIS MATERIAL IN ANY FORM FOR ANY OTHER PURPOSES SHOULD 
CONTACT THE AUTHOR VIA EMAIL.

THIS SCRIPT WAS ORIGINALLY PRODUCED IN MAPLE BY PROF MARK KELMANSON, AND
ADAPTED INTO PYTHON BY AUTHOR.

Joseph Elmes : NERC-Funded PhD Researcher in Applied Mathematics
University of Leeds : Leeds LS2 9JT : ml14je@leeds.ac.uk

Python 3.7 : Wed Aug 7 12:04:19 2019

MATH3474 : Section 1 : Lectures 1 to 4 
SCRIPT MATH3474_1.py : POLYNOMIAL INTERPOLATION
"""

import numpy as np
import matplotlib.pyplot as pt
from MATH3474 import plot_setup, Chebyshev_grid

def lagrange_interp(x_arr, X, U):
    """
    Let X = {x0, x1, ..., xn} be n+1 distinct real numbers (x0 < x1 <...< xn)
    with associated function values U = {u0, u1, ..., un}. Returns the
    Lagrangian polynomial that interpolates {X, U} on x_arr.

    Parameters
    ----------
    x_arr : numpy.ndarray
        The array of points on which to interpolate onto.
    X : numpy.ndarray
        The known points from which to interpolate.
    U : numpy.ndarray
        The associated function values of X.
    """

    assert len(X) == len(U) #both X and U must be of the same length
    n, S = len(X), 0
    for i in range(n):
        P = 1
        for j in range(n):
            if j!=i:
                P*=(x_arr-X[j])/(X[i]-X[j])
                
        S+=P*U[i]
    
    return S

def divided_diff(x_arr, X, U):
    """
    Let X = {x0, x1, ..., xn} be n+1 distinct real numbers (x0 < x1 <...< xn)
    with associated function values U = {u0, u1, ..., un}. Returns the
    nth-degree divided-difference polynomial that interpolates {X, U} on x_arr.
    
    Parameters
    ----------
    x_arr : numpy.ndarray
        The array of points on which to interpolate onto.
    X : numpy.ndarray
        The known points from which to interpolate.
    U : numpy.ndarray
        The associated function values of X.
    """

    assert len(X) == len(U) #both X and U must be of the same length
    n = len(X)
    D, B = np.zeros(n), np.copy(U)

    for i in range(1, n):
        A = np.copy(B)
        D[i-1] = B[i-1]
        for j in range(i, n):
            B[j]=(A[j]-A[j-1])/(X[j]-X[j-i])

    D[-1] = B[n-1]
    print('\nDivided differences of {X, U} on [a, b]:')
    for i in range(len(D)):
        print('\tD_{} = {}'.format(i, round(D[i], 5)))

    S, P = 0, 1
    for i in range(n):
        S+=P*D[i]
        P*=(x_arr-X[i])

    return S

def Psi_n(x_arr, X):
    """
    Polynomial \Psi_n(x) which has roots at the n + 1 nodes
    {x_0, x_1, ..., x_n}.
    
    Parameters
    ----------
    x_arr : numpy.ndarray
        The array of points on which to interpolate onto.
    X : numpy.ndarray
        The known points from which to interpolate.
    """
    P = 1
    for i in range(len(X)):
        P*=(x_arr-X[i])

    return P        

def Example_1(f, a, b, n):
    """
    Example 1.1:
    Lagrange interpolation of test function f(x) on [a, b] with 
    n + 1 nodes.

    Parameters
    ----------
    f : Function
        Test function on which to perform the Lagrange interpolation.
    a : Float
        Lower value of domain.
    b : Float
        Upper value of domain.
    n : Integer
        Number of regularly-spaced nodes in [a, b].
    """
    from sympy import lambdify, latex
    from sympy.abc import x

    g = lambdify(x, f, 'numpy') #function of symbolic expression

    fig, ax = plot_setup('$x$', scale=s)
    X = np.linspace(a, b, n+1)
    U = g(X)
    x_arr = np.linspace(a, b, 501)
    ax.plot(X, U, 'x', markersize=7*s, label='Data points')
    ax.plot(x_arr, lagrange_interp(x_arr, X, U), label = 'Lagrangian Interpolation')
    ax.plot(x_arr, g(x_arr), label='${}$'.format(latex(f)))
    ax.legend(fontsize=16*s)
    pt.show()
    
def Example_4(f, a, b, n):
    """
    Example 1.4:
    Divided-difference algorithm for f(x) on interval [a, b] with 
    n + 1 nodes.

    Parameters
    ----------
    f : Function
        Test function on which to perform the divided difference interpolation.
    a : Float
        Lower value of domain.
    b : Float
        Upper value of domain.
    n : Integer
        Number of regularly-spaced nodes in [a, b].
    """
    from sympy import lambdify, latex
    from sympy.abc import x

    g = lambdify(x, f, 'numpy') #function of symbolic expression
    
    fig, ax = plot_setup('$x$', scale=s)
    X, x_arr = np.linspace(a, b, n+1), np.linspace(a, b, 501)
    U = g(X)

    ax.plot(X, U, 'x', markersize=7*s, label='Data points')
    ax.plot(x_arr, divided_diff(x_arr, X, U),
            label="Newton's Divided Difference")
    ax.plot(x_arr, g(x_arr), label='${}$'.format(latex(f)))
    ax.legend(fontsize=16*s)
    pt.show()

def Example_6(f, a, b, n):
    """
    Example 1.6:
    Comparison of divided-difference interpolation with Lagrangian
    interpolation for f(x) on interval [a, b] with n + 1 nodes.

    Parameters
    ----------
    f : Function
        Test function on which to  compare interpolation methods.
    a : Float
        Lower value of domain.
    b : Float
        Upper value of domain.
    n : Integer
        Indicates number of regularly-spaced nodes in [a, b].
    """
    from sympy import lambdify
    from sympy.abc import x

    g = lambdify(x, f, 'numpy') #function of symbolic expression

    fig, ax = plot_setup('$x$', '$f(x)-p_n(x)$', scale=s)
    X = np.linspace(a, b, n+1)
    U, x_arr = g(X), np.linspace(a, b, 501)
    L, D = lagrange_interp(x_arr, X, U), divided_diff(x_arr, X, U)

    ax.plot(X, np.zeros(n+1), 'x', markersize=7*s,
            label='Grid points $\\{x_0,x_1,\\cdots,x_n\\}$')
    ax.plot(x_arr, g(x_arr)-D, label = "Newton's Divided Difference")
    ax.plot(x_arr, g(x_arr)-L, linestyle='-.', label='Lagragian Interpolation')
    ax.legend(fontsize=16*s)
    pt.show()

def Example_7(a, b, n):
    """
    Example 1.7:
    Comparison of \Psi_n as defined in (1.5) for when grid is regularly
    spaced and using Chebyshev points, highlighting the Runge phenomenom.

    Parameters
    ----------
    a : Float
        Lower value of domain.
    b : Float
        Upper value of domain.
    n : Integer
        Indiciates number of nodes in [a, b].
    """
    
    fig, ax = plot_setup('$x$', '$\\prod_{i=0}^n(x-x_i)$',
                         scale=s)
    X_1, X_2 = np.linspace(a, b, n+1), Chebyshev_grid(a, b, n)
    
    x = np.linspace(a, b, 501)
    ax.plot(x, Psi_n(x, X_1), label = "Regularly-Spaced Points")
    ax.plot(x, Psi_n(x, X_2), linestyle='-.', label='Chebyshev Points')
    ax.legend(fontsize=16*s)
    pt.show()
    

if __name__ == '__main__':
    s = .5 #scale of plots on screen
    from sympy import sqrt #, cos, exp, sin, tan
    from sympy.abc import x

    a, b, n = 2, 2.4, 3 #a, b, and n are parameter values as defined in notes
    f_expr = sqrt(x) # function on x in [a, b] used in the following examples

#    # Example 1.1: Lagrangian Interpolation
#    Example_1(f_expr, a, b, n)
#
#    # Example 1.4: Newton's Divided Difference Interpolation
#    Example_4(f_expr, a, b, n)
#
#    # Example 1.6: Interpolation Error
#    Example_6(f_expr, a, b, n)
#
#    # Example 1.7: Runge Phenomenom
#    Example_7(a, b, n)
