#!/usr/bin/env python3

import numpy as np
from scipy.integrate import odeint
import stelo.model_reader as mr
import stelo.helpers as h

G = 6.67428e-8
gasconst = 83144599.

def makemod(r, rho, T, g, gamma=5./3.):
    dtbardr = h.grad(r, T)
    hrho = 1. / (rho * h.grad(rho,r))
    diff = (gamma - 1.) * hrho * T - dtbardr
    np.savetxt('mymod.dat', np.array([r, rho, T, dtbardr, diff, g, hrho]).T)

def makerayleigh(r, rho, T, g, s=None, P=None):
    if P is None:
        P = rho * T * gasconst
    if s is None:
        e = 1.5 * P / rho
        s = (P / rho + e) / T
    ref_arr = np.zeros((r.shape[0],10), dtype='f8')
    ref_arr[:,0] = r
    ref_arr[:,1] = rho
    ref_arr[:,2] = h.grad(r, np.log(rho), order=1)
    ref_arr[:,3] = h.grad(r, np.log(rho), order=2)
    ref_arr[:,4] = P
    ref_arr[:,5] = T
    ref_arr[:,6] = h.grad(r, np.log(T), order=1)
    ref_arr[:,7] = h.grad(r, s, order=1)
    ref_arr[:,8] = s
    ref_arr[:,9] = g
    ref_arr = ref_arr[::-1,:]
    with open('mymod','wb') as f:
        pi_integer = np.array([314],dtype='i4')
        pi_integer.tofile(f)
        nr_ref = np.array([ref_arr.shape[0]], dtype='i4')
        nr_ref.tofile(f)
        ref_arr.T.tofile(f)

def lane_emden(Y, x, n=1.5):
    z, y = Y
    return [-y**n - 2.0 / x * z, z]

def testmod(n=500, T0=1e7, rho0=10., n_poly=1.5):
    r = np.linspace(1e8,1e10,n)
    p0 = T0 * rho0 * gasconst
    alpha = np.sqrt((n_poly + 1) * p0 / (4 * np.pi * G * rho0**2))
    xi = r / alpha
    thetaprime, theta = odeint(lane_emden, [0.,1.], xi).T
    #rho = np.exp(-(r/5e10)**2) * 40.
    rho = rho0 * theta**n_poly
    T = T0 * theta
    g = 4*np.pi * G * rho0 * alpha * thetaprime
    #T = 1e7 - 1e5 * (np.fmin(r,1e10) - 5e8) / 1e11
    #T = 1e7 - 1e5 * (r - 5e8) / 1e11
    #T = 1e7 * np.ones_like(r)
    #T = T0 * rho**(1/n_poly)
    #m = np.cumsum(4./3.*np.pi*(r[1:]**3 - r[:-1]**3) * rho[1:])
    #m = np.append(m,m[-1])
    #g = G * m / r**2
    return r, rho, T, g

def frommod():
    m = mr.mesa_model(903)
    makerayleigh(m.r, m.rho, m.T, m.grav, s=m.s, P=m.P)

#makemod(*testmod())
frommod()
#makerayleigh(*testmod())
