#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
COPYRIGHT NOTICE: THIS SCRIPT IS FOR THE SOLE USE OF STUDENTS 
ATTENDING THE MATH3474 COURSE. ANY PERSON WISHING TO COPY, USE OR 
PROCESS THIS MATERIAL IN ANY FORM FOR ANY OTHER PURPOSES SHOULD 
CONTACT THE AUTHOR VIA EMAIL.

THIS SCRIPT WAS ORIGINALLY PRODUCED IN MAPLE BY PROF MARK KELMANSON, AND
ADAPTED INTO PYTHON BY AUTHOR.

Joseph Elmes : NERC-Funded PhD Researcher in Applied Mathematics
University of Leeds : Leeds LS2 9JT : ml14je@leeds.ac.uk

Python 3.7 : Thu Aug 22 14:31:09 2019

MATH3474 : Section 3 : Lectures 18
SCRIPT MATH3474_7.py : TOEPLITZ MATRICES and CHOLESKY DECOMPOSITION
"""

import numpy as np
import matplotlib.pyplot as pt
from MATH3474 import plot_setup

def tridiag_mat(a_i, b_i, c_i, n):
    """
    Returns the tridiagonal system with values a_i, b_i and c_i (which is in
    fact a Toeplitz matrix).
    
    Parameters
    ----------
    a_i : Float
        First value of sparse tridiagonal system.
    b_i : Float
        Second value of sparse tridiagonal system which runs down the diagonal.
    c_i : Float
        Third value of sparse tridiagonal system.
    n : Integer
        Determines the size of our square sparse tridiagonal matrix.
    """
    from scipy.sparse import diags
    
    c= [a_i, b_i, c_i]
    return diags(c, [-1, 0, 1], shape=(n, n)).toarray()

def LU_decomposition(a_i, b_i, c_i, n):
    """
    Decomposes out tridiagonal matrix as the product of a lower-triangular
    matrix L* and upper-triangular matrix U*.
    
    Parameters
    ----------
    a_i : Float
        First value of sparse tridiagonal system.
    b_i : Float
        Second value of sparse tridiagonal system which runs down the diagonal.
    c_i : Float
        Third value of sparse tridiagonal system.
    n : Integer
        Determines the size of our square sparse tridiagonal matrix which we
        decompose.
    """
    a, b, c = a_i*np.ones(n), b_i*np.ones(n), c_i*np.ones(n)
    l, v, w = np.zeros(n), np.zeros(n), np.zeros(n)
    v[0], w[0] = b[0], c[0]
    for i in range(1, n):
        l[i]=a[i]/v[i-1]
        v[i] = b[i]-l[i]*w[i-1]
        w[i] = c[i]

    return l, v, w

def cholesky_solver(a_i, b_i, c_i, n, beta):
    """
    Returns the solution X to T.X = beta, where T is the Toeplitz matrix with
    diagonal entries a_i, b_i and c_i.
    
    Parameters
    ----------
    a_i : Float
        First value of sparse tridiagonal system.
    b_i : Float
        Second value of sparse tridiagonal system which runs down the diagonal.
    c_i : Float
        Third value of sparse tridiagonal system.
    n : Integer
        Determines the size of our square sparse tridiagonal matrix which we
        decompose.
    beta : numpy.ndarray
        Vector B in Ax = B, for which we solve for vector x.   
    """
    l, v, w = LU_decomposition(a_i, b_i, c_i, n) #L* and U* decomposition
    x, y = np.zeros(n), np.zeros(n)
    y[0] = beta[0]
    
    for i in range(1, n): #First solve for y in L* y = beta
        y[i] = beta[i] - l[i]*y[i-1]
        w[i] = c_i
    x[-1] = (y[-1]/v[-1])
    for i in range(2, n+1): #Then solve for x in U* x = y, where x is final sol
        x[-i] = (y[-i]-w[-i]*x[-(i-1)])/v[-i]
      
    return x

def Example_2(a_i, b_i, c_i):
    """
    Example 3.2:
    Plots the time it takes to solve an n x n tridiagonal matrix system with
    entries a_i, b_i and c_i, comparing the Cholesky factorisation method with
    the generic algorithm, i.e., inv(T)*B = X to solve for solution vector X.
    (The latter would be the general method one would utilise if the system
    matrix were not tridiagonal).
    
    Parameters
    ----------
    a_i : Float
        First value of sparse tridiagonal system.
    b_i : Float
        Second value of sparse tridiagonal system which runs down the diagonal.
    c_i : Float
        Third value of sparse tridiagonal system.
    """
    import time

    #n-values
    n_vals = np.linspace(5, 500, 495, dtype=int) #i.e., 5, 6, 7, ..., 500
    #where we shall store recorded times:
    time_vals1, time_vals2 = np.zeros(495), np.zeros(495) 
    for i, n in enumerate(n_vals): #loop through different n-values
        T = tridiag_mat(a_i, b_i, c_i, n) #Construct n x n tridiagonal matrix
        #random n-dimensional vector with integer values between 0 and 10:
        beta = np.random.randint(0, 10, n)
        start = time.time() #T_1
        cholesky_solver(a_i, b_i, c_i, n, beta)#Cholesky factrorisation method
        mid = time.time() #T_2
        np.matmul(np.linalg.inv(T), beta)#Numpy inverse (np.linalg.inv) * beta
        end = time.time() #T_3
        time_vals1[i], time_vals2[i] = mid-start, end-mid #T_2-T_1, T_3-T_2
    fig1, ax1 = plot_setup('$n$', 'Time (ms)', scale=s) #plotting environment
    ax1.plot(n_vals, time_vals1*1E3, label='Cholesky Factorisation Method')
    ax1.plot(n_vals, time_vals2*1E3, label='Numpy Inverse Method')  
    ax1.legend(fontsize=16*s)

    pt.show()

if __name__ == '__main__':
    s = .5 #scale of plots on screen
    a_i, b_i, c_i = 1, 10, 1
    print('T({}, {}, {})=\n'.format(int(a_i), int(b_i), int(c_i)),
          tridiag_mat(a_i, b_i, c_i, 100))
    
#    #Example 3.2: Solution of Sparse Matrices
#    Example_2(a_i, b_i, c_i)
#    #Is there a power law? If so, what do you think it is?