#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
COPYRIGHT NOTICE: THIS SCRIPT IS FOR THE SOLE USE OF STUDENTS 
ATTENDING THE MATH3474 COURSE. ANY PERSON WISHING TO COPY, USE OR 
PROCESS THIS MATERIAL IN ANY FORM FOR ANY OTHER PURPOSES SHOULD 
CONTACT THE AUTHOR VIA EMAIL.

THIS SCRIPT WAS ORIGINALLY PRODUCED IN MAPLE BY PROF MARK KELMANSON, AND
ADAPTED INTO PYTHON BY AUTHOR.

Joseph Elmes : NERC-Funded PhD Researcher in Applied Mathematics
University of Leeds : Leeds LS2 9JT : ml14je@leeds.ac.uk

Python 3.7 : Thu Aug 22 12:15:31 2019

MATH3474 : Section 3 : Lectures 17
SCRIPT MATH3474_6.py : MATRIX NORMS and GERSCHGORIN DISCS
"""

import numpy as np
import matplotlib.pyplot as pt
from MATH3474 import plot_setup

def one_norm(M):
    """
    Returns the one-norm of n x m matrix M.
    
    Parameters
    ----------
    M : numpy.ndarray
        n x m matrix of which we return the one-norm.
    """
    n, m = M.shape
    colsum = np.zeros(m)
    
    for j in range(m):
        colsum[j] = np.sum(np.abs(M[:, j]))
        
    return np.sort(colsum)[m-1]

def two_norm(M):
    """
    Returns the two-norm of n x n matrix M.
    
    Parameters
    ----------
    M : numpy.ndarray
        n x m matrix of which we return the two-norm.
    """
    n, m = M.shape
    assert n == m #Requires M be a square matrix
    B = np.matmul(M.T, M)
    
    return np.sqrt(rho(B))

def inf_norm(M):
    """
    Returns the infinity-norm of n x m matrix M.
    
    Parameters
    ----------
    M : numpy.ndarray
        n x m matrix of which we return the infity-norm.
    """
    n, m = M.shape
    rowsum = np.zeros(n)
    
    for j in range(n):
        rowsum[j] = np.sum(np.abs(M[j, :]))
        
    return np.sort(rowsum)[n-1]

def frobenius_norm(M):
    """
    Returns the frobenius norm of matrix M.
    
    Parameters
    ----------
    M : numpy.ndarray
        n x m matrix of which we return the Frobenius norm.
    """
    n, m = M.shape
    S = 0
    
    for i in range(m):
        for j in range(n):
            S += M[j, i]**2 #S is the sum of all matrix elements squared
            
    return np.sqrt(S)

def eig_vals(M):
    """
    Returns the eigenvalues of square matrix M. This function is
    essentially a wrapper for the numpy function np.linalg.eig. Note that
    on the return line, writing instead np.linalg.eig(M)[1] would return
    the eigenvectors.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we return the eigenvalues.
    """
    n, m = M.shape
    assert n == m #Requires M be a square matrix
    
    return np.linalg.eig(M)[0]

def rho(M):
    """
    Returns the spectral radius of square matrix M.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix of which we return the spectral radius.
    """
    n, m = M.shape
    assert n == m #Requires M be a square matrix
    eigvals = eig_vals(M)
    
    modlam = np.array([np.abs(eigval) for eigval in eigvals])
    
    return np.sort(modlam)[n-1]

def plot_circle(ax, x0, y0, r, c='k', a=1):
    """
    Plots the circle centred at (x0, y0) with radius r on axes class, ax,
    with color c.
    
    Parameters
    ----------
    ax : matplotlib.figure.Figure
        Axes figure on which we plot the circle.
    x0 : Float
        Horizontal coordinate of the circle centre.
    y0 : Float
        Vertical coordinate of the circle centre.
    r : Float
        Radius of circle.
    c : String
        Colour of circle. Default is 'k', which indicates black.
    a = Float
        Transparency/alpha of the plotted circle.
    """
    from math import pi

    theta = np.linspace(0, 2*pi, 200)
    ax.plot(x0+r*np.cos(theta), y0+r*np.sin(theta),
            color=c, alpha=a, linewidth=s)

def Example_1(M):
    """
    Example 3.1:
    Illustration of Brauer's Theorem, showing that the eigenvalues of matrix
    M are all contained within the union of the Gerschgorin disks.
    
    Parameters
    ----------
    M : numpy.ndarray
        Square matrix which we use to illustrate Brauer's Theorem.
    """
    n, m = M.shape
    assert n==m #Ensure M is a square matrix

    fig1, ax1 = plot_setup('Re', 'Im', title='Row Disks', scale=s)
    fig2, ax2 = plot_setup('Re', 'Im', title='Column Disks', scale=s)
    ax1.set_aspect('equal', adjustable='box')
    ax2.set_aspect('equal', adjustable='box')
    colors = ['r', 'b', 'g']

    #Plot Eigenvalues:
    eigvals = eig_vals(M)
    lab = 'Eigenvalues'
    for eigval in eigvals:
        ax1.plot(np.real(eigval), np.imag(eigval), 'ko', markersize=7*s,
                 label=lab)
        ax2.plot(np.real(eigval), np.imag(eigval), 'ko', markersize=7*s,
                 label=lab)
        lab = '__nolegend__' #prevents duplicated legend from loop
    
    #Plot Gerschgorin disks:
    for i in range(n):
        r1 = one_norm(M[i,:].reshape((3, 1)))-abs(M[i,i])
        r2 = inf_norm(M[:,i].reshape((1, 3)))-abs(M[i,i])
        plot_circle(ax1, M[i, i], 0, r1, c=colors[i])
        plot_circle(ax2, M[i, i], 0, r2, c=colors[i])
        for ax in [ax1, ax2]:
            ax.plot(M[i,i], 0, 'o', fillstyle='none', color=colors[i],
                    markersize=7*s)
    
    ax1.legend(fontsize=16*s)
    ax2.legend(fontsize=16*s)
    pt.show()
        
if __name__ == '__main__':
    s = .5 #scale of plots on screen
    d = 3 #Printed precision of calculated values
    A = np.array([
            [-4, 3, 3],
            [1, -2, 1],
            [0, 1, -5]
            ])

    print('One-norm: {}'.format(round(inf_norm(A), d)))
    print('Two-norm: {}'.format(round(two_norm(A), d)))
    print('Infinity-norm: {}'.format(round(one_norm(A), d)))
    print('Frobenius norm: {}'.format(round(frobenius_norm(A), d)))
    print('Spectral radius: {}'.format(round(rho(A), d)))
    
#    #Example 3.1: Illustration of Brauer's Theorem
#    Example_1(A)