#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
COPYRIGHT NOTICE: THIS SCRIPT IS FOR THE SOLE USE OF STUDENTS 
ATTENDING THE MATH3474 COURSE. ANY PERSON WISHING TO COPY, USE OR 
PROCESS THIS MATERIAL IN ANY FORM FOR ANY OTHER PURPOSES SHOULD 
CONTACT THE AUTHOR VIA EMAIL.

THIS SCRIPT WAS ORIGINALLY PRODUCED IN MAPLE BY PROF MARK KELMANSON, AND
ADAPTED INTO PYTHON BY AUTHOR.

Joseph Elmes : NERC-Funded PhD Researcher in Applied Mathematics
University of Leeds : Leeds LS2 9JT : ml14je@leeds.ac.uk

Python 3.7 : Tue Aug 20 16:21:58 2019

MATH3474 : Section 2 : Lectures 15 to 16
SCRIPT MATH3474_5.py : MEHRSTELLENVERFAHREN: COMPACT MOLECULES
"""

import numpy as np
import matplotlib.pyplot as pt
from MATH3474 import plot_setup
from math import pi

def symb_Laplace_operator(u):
    return simplify(diff(u, x, 2)+diff(u, y, 2))

def five_point_func(f, u, plus, cross, x0, y0, h_vals):
    """
    Solves for u(x0, y0) in the the poisson equationΔu=f(x,y) and f specified
    using a five-point stencil with uniform-grid spacing as determined by
    h_vals.
    
    Parameters
    ----------
    f : Function
        Function f as defined in Poisson's equation Δu=f(x,y).
    u : Function
        Function u as defined in Poisson's equation Δu=f(x,y).
    plus : numpy.ndarray
        The plus stencil centered at (x0, y0) for h in h_vals.
    cross : numpy.ndarray
        The plus stencil centered at (x0, y0) for h in h_vals.
    x0 : Float
        The x-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    y0 : Float
        The y-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    h_vals : numpy.ndarray
        The values of uniform grid-spacing for which we wish to solve for u in
        Δu=f(x,y), for f(x, y) specified in problem.
    """
    from MATH3474 import newton_raphson
    vals = np.zeros(len(h_vals))
    u_val, f0 = u(x0, y0), f(x0, y0)
    
    def LHS(u0, i):
        return (plus[i]-4*u0)/h_vals[i]**2
    
    RHS = f0
    
    for i, h in enumerate(h_vals):
        g = lambda u0: LHS(u0, i)-RHS
        vals[i] = newton_raphson(g, u_val)
        
    return vals

def five_X_point_func(f, u, plus, cross, x0, y0, h_vals):
    """
    Solves for u(x0, y0) in the the poisson equationΔu=f(x,y) and f specified
    using a five-point X stencil with uniform-grid spacing as determined by
    h_vals.

    Parameters
    ----------
    f : Function
        Function f as defined in Poisson's equation Δu=f(x,y).
    u : Function
        Function u as defined in Poisson's equation Δu=f(x,y).
    plus : numpy.ndarray
        The plus stencil centered at (x0, y0) for h in h_vals.
    cross : numpy.ndarray
        The plus stencil centered at (x0, y0) for h in h_vals.
    x0 : Float
        The x-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    y0 : Float
        The y-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    h_vals : numpy.ndarray
        The values of uniform grid-spacing for which we wish to solve for u in
        Δu=f(x,y), for f(x, y) specified in problem.
    """
    from MATH3474 import newton_raphson
    vals = np.zeros(len(h_vals))
    u_val, f0 = u(x0, y0), f(x0, y0)
    
    def LHS(u0, i):
        return (cross[i]-4*u0)/(2*h_vals[i]**2)
    
    RHS = f0
    
    for i, h in enumerate(h_vals):
        g = lambda u0: LHS(u0, i)-RHS
        vals[i] = newton_raphson(g, u_val)
        
    return vals

def nine_point_func(f, u, plus, cross, x0, y0, h_vals):
    """
    Solves for u(x0, y0) in the the poisson equationΔu=f(x,y) and f specified
    using a nine-point stencil with uniform-grid spacing as determined by
    h_vals.

    Parameters
    ----------
    f : Function
        Function f as defined in Poisson's equation Δu=f(x,y).
    u : Function
        Function u as defined in Poisson's equation Δu=f(x,y).
    plus : numpy.ndarray
        The plus stencil centered at (x0, y0) for h in h_vals.
    cross : numpy.ndarray
        The plus stencil centered at (x0, y0) for h in h_vals.
    x0 : Float
        The x-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    y0 : Float
        The y-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    h_vals : numpy.ndarray
        The values of uniform grid-spacing for which we wish to solve for u in
        Δu=f(x,y), for f(x, y) specified in problem.
    """
    from MATH3474 import newton_raphson
    vals = np.zeros(len(h_vals))
    u_val, f0 = u(x0, y0), f(x0, y0)
    
    def LHS(u0, i):
        return (cross[i]+4*plus[i]-20*u0)/(6*h_vals[i]**2)
    
    RHS = f0
    
    for i, h in enumerate(h_vals):
        g = lambda u0: LHS(u0, i)-RHS
        vals[i] = newton_raphson(g, u_val)
        
    return vals

def nine_point_mod_func(f, u, plus, cross, x0, y0, h_vals):
    """
    Solves for u(x0, y0) in the the poisson equationΔu=f(x,y) and f specified
    using the nine-point stencil “Mehrstellenverfahren” with uniform-grid
    spacing as determined by h_vals.

    Parameters
    ----------
    f : Function
        Function f as defined in Poisson's equation Δu=f(x,y).
    u : Function
        Function u as defined in Poisson's equation Δu=f(x,y).
    plus : numpy.ndarray
        The plus stencil centered at (x0, y0) for h in h_vals.
    cross : numpy.ndarray
        The plus stencil centered at (x0, y0) for h in h_vals.
    x0 : Float
        The x-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    y0 : Float
        The y-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    h_vals : numpy.ndarray
        The values of uniform grid-spacing for which we wish to solve for u in
        Δu=f(x,y), for f(x, y) specified in problem.
    """
    from MATH3474 import newton_raphson
    vals = np.zeros(len(h_vals))
    u_val, f0 = u(x0, y0), f(x0, y0)
    
    def LHS(u0, i):
        return (cross[i]+4*plus[i]-20*u0)/(6*h_vals[i]**2)
    f_plus = f(x0+h_vals, y0)+f(x0, y0+h_vals)+f(x0-h_vals, y0)+\
                        f(x0, y0-h_vals)
    
    for i, h in enumerate(h_vals):
        RHS = (f_plus[i]+8*f0)/12
        g = lambda u0: LHS(u0, i)-RHS
        vals[i] = newton_raphson(g, u_val)
        
    return vals

def Example_13(f_symb, u_symb, x0, y0):
    """
    example 2.13
    Compares the attenuation errors where solving for u(x0, y0) in the 
    poisson equationΔu=f(x,y) with f specified.

    Parameters
    ----------
    f_symb : sympy.core.function
        Symbolic representation of function f(x, y) in Δu=f(x,y).
    u_symb : sympy.core.function
        Symbolic representation of function u(x, y) in Δu=f(x,y). This is
        the exact analytical solution.
    x0 : Float
        The x-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    y0 : Float
        The y-value at which we attempt to evaluate u in Δu=f(x,y) for f(x, y)
        specified.
    """
    from sympy import lambdify

    f = lambdify((x, y), f_symb, "numpy")
    u = lambdify((x, y), u_symb, "numpy")
    u_true = u(x0, y0)
    labels = ["5-point $+$", "5-point $\\times$", "9-point",
              "Modified 9-point"]
    funcs = [five_point_func, five_X_point_func, nine_point_func,
                             nine_point_mod_func]

    #Consider base b closer to 1 and n much larger to increase resolution:
    b, n = 2, 8 #b, n = 1.01, 300

    h_vals = b**np.linspace(0, -n, n+1)

    #construct models:
    plus = u(x0+h_vals, y0)+u(x0, y0+h_vals)+u(x0-h_vals, y0)+u(x0, y0-h_vals)
    cross = u(x0+h_vals, y0+h_vals)+u(x0-h_vals, y0+h_vals)+\
                        u(x0-h_vals, y0-h_vals)+u(x0+h_vals, y0-h_vals)
    
    fig, ax = plot_setup('$h$', '$\\alpha(h)$'.format(b),
            x_log=True, y_log=True, bx=b, by=b, scale=s)

    for func, lab in zip(funcs, labels):
        U = func(f, u, plus, cross, x0, y0, h_vals)
        alpha = (U[:-1]-u_true)/(b**2*(U[1:]-u_true))
        ax.plot(h_vals[:-1], alpha, label=lab)

    ax.legend(fontsize=16*s)
    pt.show()
        
if __name__ == '__main__':
    s = .5 #scale of plots on screen
    from sympy import sin, cos, tan, exp, sqrt, diff, simplify
    from sympy.abc import x, y
    
    #Solving Poisson's equation at (x,y)=(x0,y0),
    #whereby Δu(x,y)=f(x,y):
    u_expr = sin(x)*exp(y**2) #Essentially, one chooses exact solution first
    f_expr = symb_Laplace_operator(u_expr) #Laplace operator
    x0, y0 = 0.5, 0.5 #values at which we solve for u(x, y)
    
#    #Example 2.13: An application of Mehrstellenverfahren
    Example_13(f_expr, u_expr, x0, y0)