#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
COPYRIGHT NOTICE: THIS SCRIPT IS FOR THE SOLE USE OF STUDENTS 
ATTENDING THE MATH3474 COURSE. ANY PERSON WISHING TO COPY, USE OR 
PROCESS THIS MATERIAL IN ANY FORM FOR ANY OTHER PURPOSES SHOULD 
CONTACT THE AUTHOR VIA EMAIL.

THIS SCRIPT WAS ORIGINALLY PRODUCED IN MAPLE BY PROF MARK KELMANSON, AND
ADAPTED INTO PYTHON BY AUTHOR.

Joseph Elmes : NERC-Funded PhD Researcher in Applied Mathematics
University of Leeds : Leeds LS2 9JT : ml14je@leeds.ac.uk

Python 3.7 : Wed Aug 7 12:13:58 2019
"""

def plot_setup(x_label='', y_label='', x_log=False, y_log=False, bx=10, by=10,
               scale=1, title='', dpi=100):
    """
    Sets up figure environment.

    Parameters
    ----------
    x_label : String
        Label of the x axis.
    y_label : String
        Label of the y axis.
    x_log : Boolean
        Determines whether the x axis is log scale.
    y_log : Boolean
        Determines whether the y axis is log scale.
    bx : Float
        The base of the x axis if x_log == True.
    by : Float
        The base of the y axis if y_log == True.
    scale : Float
        Scale of figure environment with respect to an aspect ratio of 13.68 
        x 7.68.
    title : String
        Title of figure plot.
    dpi : Integer
        Density of pixels per square inch of figure plot.
    """
    import matplotlib.pyplot as pt
    import matplotlib as mpl

    mpl.style.use('seaborn-ticks')
    fig, ax = pt.subplots(figsize=(13.68*scale, 7.68*scale), dpi=dpi)
    pt.xticks(fontsize=scale*16)
    pt.yticks(fontsize=scale*16)
    pt.xlabel(x_label, fontsize=scale*20)
    pt.ylabel(y_label, fontsize=scale*20)
    pt.title(title, fontsize=scale*22)
    if x_log: pt.xscale('log', basex=bx)
    if y_log: pt.yscale('log', basey=by)
    pt.grid(linewidth=0.5)

    return fig, ax

def Chebyshev_grid(a, b, n):
    """
    Creates a 1D grid on [a, b] corresponding to n+1 Chebyshev points.

    Parameters
    ----------
    a : Float
        Beginning value of domain.
    b : Float
        End value of domain.
    n : Integer
        Indicates the number of Chebyshev points.
    """
    import numpy as np
    theta, middle, radius = np.pi/n, (a+b)/2, (b-a)/2

    return np.array([middle-radius*np.cos(i*theta) for i in range(n+1)])

def T_n(n, y):
    """
    Returns the first n+1 evaluations of Chebyshev polynomials of the first
    kind on y.

    Parameters
    ---------- 
    n : Integer
        Indicates the n-th Chebyshev polynomial of the first kind.
    y : numpy.ndarray
        The grid on which to perform the evaluation of the Chebyshev
        polynomial.
    """
    import numpy as np

    T = np.zeros((1, len(y)))
    
    for i in range(n+1):
        if i == 0:
            T[0,:] = 1
        elif i == 1:
            T = np.append(T, [y], axis=0)
        else:
            T = np.append(T, [2*y*T[-1] - T[-2]], axis=0)

    return T

def T_n_func(n):
    """
    Returns the numpy function of the n-th Chebyshev polynomial of the first
    kind.
    
    Parameters
    ---------- 
    n : Integer
        Indicates the function of n-th Chebyshev polynomial of the first kind
        to return.
    """
    import numpy as np
    Tn_coeff = np.zeros(n+1)
    Tn_coeff[-1] = 1
    
    return np.polynomial.chebyshev.Chebyshev(Tn_coeff)

def U_n(n, y):
    """
    Returns the numpy function of the n-th Chebyshev polynomial of the second
    kind.

    Parameters
    ---------- 
    n : Integer
        Indicates the n-th Chebyshev polynomial of the second kind.
    y : numpy.ndarray
        The grid on which to perform the evaluation of the Chebyshev
        polynomial.
    """

    import scipy
    import numpy as np
    Un_coeff = np.zeros(n+1)
    Un_coeff[-1] = 1
    
    return scipy.special.eval_chebyu(Un_coeff)

def p_n(coeff_vec, x):
    """
    Returns the power series evaluation of x with integer powers indicated by
    the coefficients in coeff_vec.

    Parameters
    ---------- 
    coeff_vec : numpy.ndarray
        Coefficients of powers series.
    x : numpy.ndarray
        Values at which to evaluate the power series.
    """
    S = 0
    for i, a in enumerate(coeff_vec):
        S+=a*(x**i)
    
    return S

def p_n_der(coeff_vec, x):
    
    S = 0
    for i, a in enumerate(coeff_vec):
        if i != 0:
            S+=i*a*(x**(i-1))

    return S

def p2_norm(v):
    """
    Evaluates the p-2 norm (also known as the Euclidean norm) of vector v.

    Parameters
    ---------- 
    v : numpy.ndarray
        The vector on which to perform the p-2 norm.
    """
    import numpy as np
    return np.sqrt(np.matmul(v, v))

def pInf_norm(v):
    """
    Evaluates the p-infinity norm of vector v.
    
    Parameters
    ---------- 
    v : numpy.ndarray
        The vector on which to perform the p-infinity norm.
    """
    import numpy as np
    return np.max(np.abs(v))

def trapezium_rule(f, a, b, n):
    """
    Integrates the function f between a and b with n + 1 grid points using the
    trapezium rule.

    Parameters
    ---------- 
    f : Function
        The function to integrate.
    a : Float
        The lower limit of the definite integral.
    b : Float
        The upper limit of the definite integral.
    n : Integer
        The number of sections on which to perform the trapezium rule.
    """
    S, h = (f(a)+f(b))/2, (b-a)/n

    for i in range(1, n):
        S+=f(a+i*h)
    
    return h*S

def newton_raphson(f, init_val, sup_output=True, max_iteration=250):
    """
    Newton--Raphson method which utilises the Secant-Method to
    approximate the derivative of function f. The result should return the#
    closest root of f to init_val.

    Parameters
    ---------- 
    f : Function
        The function of which the Newton--Raphson method will attempt find the
        root.
    init_val : Float
        The initial guess of the Newton--Raphson method.
    sup_output : Boolean
        This will determine whether to print or not the various adaptions,
        such as whether the maximum number of iterations has been reached. The
        default is False.
    max_iteration : Integer
        Maximum number of iterations of the Newton-Raphson method. Default is
        250.
    """
    tolerance, F, eps = 1E-15, f(init_val), 1E-8
    error, iteration = abs(F), 1
    val = init_val

    while error > tolerance:
        f_prime = (f(val+eps/2)-f(val-eps/2))/eps

        val -= F/f_prime
        F = f(val)
#        print(F)
        error = abs(F)
        iteration +=1

        if iteration > max_iteration:
#            print(F)
            if not sup_output:
                print('Could not converge to machine precision within {} \
#iterations.\n'.format(max_iteration))
            return val
    
    return val

def x_to_t(x_val, a, b):
    """
    Returns the linear mapping of x_val from [a, b] to [-1, 1].

    Parameters
    ---------- 
    x_val : numpy.ndarray
        The values on which to perform the mapping.
    a : Float
        The lower limit of the domain on which to perform the linear mapping.
    b : Float
        The upper limit of the domain on which to perform the linear mapping.
    """
    return (2*x_val-(b+a))/(b-a)

def t_to_x(t_val, a, b):
    return ((b-a)*t_val+(b+a))/2

def root_sqr_mean_err(f1, f2, a, b):
    """
    Returns the linear mapping of t_val from [-1, 1] to [a, b].

    Parameters
    ---------- 
    t_val : numpy.ndarray
        The values on which to perform the mapping.
    a : Float
        The lower limit of the domain on which to perform the linear mapping.
    b : Float
        The upper limit of the domain on which to perform the linear mapping.
    """
    import numpy as np
    from math import pi
    g = lambda theta: (f1(np.cos(theta))-f2(np.cos(theta)))**2
    I = trapezium_rule(g, 0, pi, int(5E3))
    return np.sqrt(I)/np.sqrt(b-a)

def bisection_method(f, v1, v2, tol=1E-16):
    """
    Returns the root of function f in the domain [v1, v2] using the bisection
    method with a tolerance of tol.

    Parameters
    ---------- 
    f : Function
        The function on which to perform the bisection method, seeking roots.
    v1 : Float
        The lower limit of the domain on which to perform the bisection method.
    v2 : Float
        The upper limit of the domain on which to perform the bisection method.
    tol : Float
        The tolerance of the bisection method. The default is machine-precision.
    """
    c = (v1+v2)/2
    while (v2-v1)/2 > tol:
        if f(c) == 0:
            return c
        elif f(v1)*f(c) < 0:
            v2 = c
        else :
            v1 = c
        c = (v1+v2)/2

    return c